﻿using GeoUK.Coordinates;
using GeoUK.Ellipsoids;
using GeoUK.Projections;
using System;
using System.Collections.Generic;
using System.Diagnostics;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Windows;
using System.Windows.Controls;
using System.Windows.Data;
using System.Windows.Documents;
using System.Windows.Input;
using System.Windows.Media;
using System.Windows.Media.Imaging;
using System.Windows.Navigation;
using System.Windows.Shapes;

namespace CoordinateConverter_WPF
{
/// <summary>
/// Interaction logic for MainWindow.xaml
/// </summary>
public partial class MainWindow : Window
    {
        private string clipboardText;
        private Airy1830 airy1830 = new Airy1830();
        private BritishNationalGrid bng = new BritishNationalGrid();
        private Wgs84 wgs = new Wgs84();
        List<string> eastingsAndNorthingsList = new List<string>();

        public MainWindow()
        {
            InitializeComponent();

            EastingsAndNorthings.Foreground = new SolidColorBrush(Color.FromArgb(255, 180, 180, 180));
            LatsandLongs.Foreground = new SolidColorBrush(Color.FromArgb(255, 180, 180, 180));

            // if the user pastes a lot of data into the input box the UI becomes unresponsive
            DataObject.AddPastingHandler(EastingsAndNorthings, OnPaste);
        }

        private void OnPaste(object sender, DataObjectPastingEventArgs args)
        {
            clipboardText = Clipboard.GetText();
            eastingsAndNorthingsList = clipboardText.Split('\n').ToList();
            if (eastingsAndNorthingsList.Count>1000)
            {
                EastingsAndNorthingWarning.Text = "All " + eastingsAndNorthingsList.Count + " lines will be converted. Only the first 1000 are shown.";
            }
            else
            {
                EastingsAndNorthingWarning.Text = "";
            }
            string displayInputText = string.Join(null, eastingsAndNorthingsList.GetRange(0, Math.Min(eastingsAndNorthingsList.Count, 1000)));
            EastingsAndNorthings.Text = displayInputText;

            // we don't actually want to paste all that data into the box
            args.CancelCommand();
        }



        // as per http://johnnewcombe.net/geouk
        // license is LGPL
        private async Task<LatitudeLongitude> EastingNorthingToLatLong(double easting, double northing)
        {
            LatitudeLongitude wgsLatLong = null;
            var cartesian = GeoUK.Convert.ToCartesian(airy1830, bng, new EastingNorthing(easting, northing));
            var wgsCartesian = GeoUK.Transform.Osgb36ToEtrs89(cartesian); //ETRS89 is effectively WGS84 
            wgsLatLong = GeoUK.Convert.ToLatitudeLongitude(wgs, wgsCartesian);
            return wgsLatLong;
        }

        private async void doConversion()
        {
            List<string> latLongList = new List<string>();

            // set up progress bar
            int progressCount = 0;
            progress.Maximum = eastingsAndNorthingsList.Count;
            progress.Minimum = 0;

            foreach (var eastingAndNorthing in eastingsAndNorthingsList)
            {
                if (eastingAndNorthing.Length > 5)
                {
                    var splitLine = eastingAndNorthing.Split(',');
                    var easting = double.Parse(splitLine[0]);
                    var northing = double.Parse(splitLine[1]);
                    
                    // run on another thread so as not to block the UI
                    var latlong = await Task.Run(() => EastingNorthingToLatLong(easting, northing));
                    latLongList.Add(latlong.Latitude + "," + latlong.Longitude);
                }
                progressCount++;
                if (progressCount % 1000 == 0)
                {
                    progress.Value = progressCount;
                    progressCounter.Text = progressCount + "/" + eastingsAndNorthingsList.Count;
                }
            }
            progress.Value = progressCount;
            progressCounter.Text = progressCount + "/" + eastingsAndNorthingsList.Count;

            string outputText = string.Join(Environment.NewLine, latLongList);
            Clipboard.SetText(outputText);

            MessageBox.Show("All " + progressCount + " results have been copied to the clipboard"); ;

            string displayOutputText = string.Join(Environment.NewLine, latLongList.GetRange(0, Math.Min(latLongList.Count, 1000)));
            LatsandLongs.Text = displayOutputText;

            if (latLongList.Count > 1000)
            {
                LatsAndLongsWarning.Text = "All " + latLongList.Count + " lines were converted and saved to the clipboard. The first 1000 are shown.";
            }
            else
            {
                LatsAndLongsWarning.Text = "";
            }
        }

        private void ConvertButton_Click(object sender, RoutedEventArgs e)
        {
            doConversion();
        }

        private void Hyperlink_RequestNavigate(object sender, RequestNavigateEventArgs e)
        {
            Process.Start(new ProcessStartInfo(e.Uri.AbsoluteUri));
            e.Handled = true;
        }

        private void EastingsAndNorthings_GotFocus(object sender, RoutedEventArgs e)
        {
            if (EastingsAndNorthings.Text == "Paste comma-seperated Eastings and Northings here.")
            {
                EastingsAndNorthings.Text = "";
            }
            EastingsAndNorthings.Foreground = new SolidColorBrush(Color.FromArgb(255, 0, 0, 0));
        }

        private void LatsandLongs_GotFocus(object sender, RoutedEventArgs e)
        {
            if (LatsandLongs.Text == "Press convert and comma-seperated lats and longs will appear here.")
            {
                LatsandLongs.Text = "";
            }
            LatsandLongs.Foreground = new SolidColorBrush(Color.FromArgb(255, 0, 0, 0));

        }

        private void LatsandLongs_LostFocus(object sender, RoutedEventArgs e)
        {
            if (LatsandLongs.Text.Length == 0)
            {
                LatsandLongs.Foreground = new SolidColorBrush(Color.FromArgb(255, 180, 180, 180));
                LatsandLongs.Text = "Paste comma-seperated Eastings and Northings here.";
            }
            LatsandLongs.Foreground = new SolidColorBrush(Color.FromArgb(255, 180, 180, 180));
        }

        private void EastingsAndNorthings_LostFocus(object sender, RoutedEventArgs e)
        {
            if (EastingsAndNorthings.Text.Length == 0)
            {
                EastingsAndNorthings.Foreground = new SolidColorBrush(Color.FromArgb(255, 180, 180, 180));
                EastingsAndNorthings.Text = "Press convert and comma-seperated lats and longs will appear here.";
            }
            EastingsAndNorthings.Foreground = new SolidColorBrush(Color.FromArgb(255, 180, 180, 180));
        }
    }
}
